<?php

namespace ISProject\Helper\FileUpload;

use ISProject\Helper\Utility\Utility;

Class FileUpload
{

	/**
	 *
	 * FInfo kurulu mu?
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var boolean $fInfo
	 *
	 */
	private $fInfo;

	/**
	 *
	 * Uploads klasörüne eklenecek resimler ay, yıl formatında
	 * klasörlere ayrılsın mı?
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var boolean $uploadDateFolder
	 *
	 */
	private $uploadDateFolder;

	/**
	 *
	 * Dosyalar nereye yüklenecek?
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var string $uploadFolder
	 *
	 */
	private $uploadFolder;

	/**
	 *
	 * Kullanıcı Tarafından Yüklenebilecek en yüksek dosya boyutu KB
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var int $maxUploadFileSize
	 *
	 */
	private $maxUploadFileSize;

	/**
	 *
	 * Yüklenecek dosya adının random olup olmadığını belirtir.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var boolean $randomName
	 *
	 */
	private $randomName;

	/**
	 *
	 * Resim boyutlandırma formatları
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var array $imageResize
	 *
	 */
	private $imageResize;

	/**
	 *
	 * Sınıfın standart ayarlarını uygular.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 *
	 */
	public function __construct()
	{
		$this->checkFInfo();
		$this->setUploadFolder(CMS_UPLOAD_FOLDER);
		$this->setUploadDateFolder(true);
		$this->setRandomName(false);
		$this->setImageResize(false);
		$this->setMaxUploadFileSize(1024 * 1024 * 3);
	}

	/**
	 *
	 * Sistemdeki FInfo Sınfırını kontrol eder.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 *
	 */
	public function checkFInfo()
	{
		$this->fInfo = class_exists('finfo');
	}


	/**
	 *
	 * Resimlerin yükleneceği klasör yolunu ayarlar.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $uploadFolder
	 *
	 */
	public function setUploadFolder($uploadFolder)
	{
		$this->uploadFolder = $uploadFolder;		
	}

	/**
	 *
	 * Resimlerin yükleneceği klasör yolunu ayarlar.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param boolean $type
	 *
	 */
	public function setUploadDateFolder($type)
	{
		$this->uploadDateFolder = $type === true;
	}

	/**
	 *
	 * Yüklenecek dosya adının random olup olmadığını değiştirir
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param boolean $type
	 *
	 */
	public function setRandomName($type)
	{
		$this->randomName = $type === true;		
	}

	/**
	 *
	 * Resim boyutlandırma formatları ayarlar.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param array $size
	 *
	 */
	public function setImageResize($sizeFormats)
	{
		$this->imageResize = is_array($sizeFormats) ? $sizeFormats : false;
	}

	/**
	 *
	 * Sisteme yüklenecek en büyük dosya boyutunu belirler.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param int $size
	 *
	 */
	public function setMaxUploadFileSize($size)
	{
		$maxSize = $this->fileUploadMaxSize();
		if ($size >= $maxSize) {
			$size = $maxSize - 1024;
		}

		$this->maxUploadFileSize = $size;
	}

	/**
	 *
	 * Dosyaların yükleneceği geçerli klasör
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @return string 
	 *
	 */
	public function getUploadFolder()
	{

		if (!$this->uploadDateFolder) {
			return $this->uploadFolder;
		}

		$nowMonth = date('m');
		$nowYear  = date('Y');

		if(!file_exists($this->uploadFolder)){
			mkdir($this->uploadFolder, 0777);
		}

		if(!file_exists($this->uploadFolder . $nowYear . '/')){
			mkdir($this->uploadFolder . $nowYear . '/', 0777);
		}

		if(!file_exists($this->uploadFolder . $nowYear . '/' . $nowMonth . '/')){
			mkdir($this->uploadFolder . $nowYear . '/' . $nowMonth . '/', 0777);
		}
		return $this->uploadFolder . $nowYear . '/' . $nowMonth . '/';
	}

	/**
	 *
	 * Dosyaların yükleneceği klasörün tam yolunu döner.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $degisken
	 * @return string 
	 *
	 */
	public function getUploadPath()
	{
		$utility = new Utility();
		return $utility->getPublicFolder() . $this->getUploadFolder();		
	}


	/**
	 *
	 * Dosyanın MimeType'ını döndürür
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $filename
	 * @return string 
	 *
	 */
	public function mimeContentType($filename)
	{
	    $result = new \finfo();
        return $result->file($filename, FILEINFO_MIME_TYPE);
	}

	/**
	 *
	 * Sisteme En fazla yüklenebilecek dosya boyutunu döner.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @return int 
	 *
	 */
    public function fileUploadMaxSize()
    {
      	
		$max_size   = $this->parseInisize(ini_get('post_max_size'));
		$upload_max = $this->parseInisize(ini_get('upload_max_filesize'));

    	if ($upload_max > 0 && $upload_max < $max_size){
      		$max_size = $upload_max;
    	}

	  	return $max_size;

	}

	/**
	 *
	 * İni dosyasından gelen boyutları int olarak döndürür.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $size
	 * @return int 
	 *
	 */
	protected function parseInisize($size)
	{

		$unit = preg_replace('/[^bkmgtpezy]/i', '', $size);
		$size = preg_replace('/[^0-9\.]/', '', $size);

		if ($unit){
		    return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
		}

		return round($size);
	}

	/**
	 *
	 * Rastgele dosya adı üretir.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $name
	 * @param int $size
	 * @param string $tmp_name
	 * @return string 
	 *
	 */
	public function randomFileName($name, $size, $tmp_name, $extension)
	{
		return md5($name . $size . $tmp_name . $extension . uniqid() . microtime(1) . rand(0 ,99999));
	}	

	/**
	 *
	 * Dosyanın adını temiz ve benzersiz olarak döndürür.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $name
	 * @param string $mimeType
	 * @return string 
	 *
	 */
	protected function filename($name, $extension)
	{
		$fileName = explode('.', strtolower($name));
		$fileName = trim(preg_replace("/[^a-zA-Z0-9-]+/", "", $fileName[0]));
		if (empty($fileName)) {
			return "";
		}

		$filePath = $this->getUploadPath() . $fileName;

		$counter = 0;
		$files = glob($filePath . '*-?.' . $extension);
		if (count($files) > 0) {
			$lastFile = str_replace($filePath, '', end($files));
			$lastFile = explode('-', $lastFile);
			$counter  = (int) str_replace('.' . $extension, '', $lastFile[1]);
		}

		while (true) 
		{
			$file = $filePath . '-' . (++$counter) . '.' . $extension;
			if (!file_exists($file)) 
			{
				return $fileName . '-' . $counter;
			}
		}
	}

	/**
	 *
	 * Resim silme işlemini başlatır.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param int $imageData
	 * @return boolean 
	 *
	 */
	public function deleteImage($imageData)
	{

		if (is_object($imageData)) {
			$imageData = array(
				'path'   => $imageData->path,
				'images' => (array)$imageData->images,
			);
		}

		if (!isset($imageData['path']) || !isset($imageData['images'])) {
			return false;
		}

		$utility = new Utility();
		foreach ($imageData['images'] as $imageName) 
		{
			$imagePath = $utility->getImagePath($imageData['path'], $imageName);
			if (file_exists($imagePath)) {
				unlink($imagePath);
			}
		}

		return true;
	}

	/**
	 *
	 * Dosyanın eklenip eklenmediğini kontrol eder.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $fileName
	 * @return boolean 
	 *
	 */
	public function checkFile($fileName)
	{
		if (!isset($_FILES[$fileName])) {
			return false;
		}

		return !($_FILES[$fileName]['error'] == UPLOAD_ERR_NO_FILE);
	}

	/**
	 *
	 * Resim yükleme işlemini başlatır.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $uploadName
	 * @return array 
	 *
	 */
	public function uploadImage($uploadName)
	{

		if (!isset($_FILES[$uploadName])) {
			return array('error' => false, 'msg' => 'Input Name Bulunamadı.');
		}

		return $this->upload($_FILES[$uploadName], array('jpg' => 'image/jpg', 'jpeg' => 'image/jpeg', 'png' => 'image/png'));
	}

	/**
	 *
	 * 
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $degisken
	 * @return string 
	 *
	 */
	protected function upload($file, $mimetypes)
	{

		if (!$this->fInfo) {
			return array('error' => true, 'file' => $file, 'msg' => 'Sunucu Yapılandırmasında <b>finfo</b> uzantısı eksik.');
		}

		if ($file['error'] == UPLOAD_ERR_NO_FILE) {
			return array('error' => true, 'file' => $file, 'msg' => 'Lütfen dosya seçiniz.');
		}

		if($file['error'] != UPLOAD_ERR_OK || empty($file['name'])){
			return array('error' => true, 'file' => $file, 'msg' => 'Dosya Yüklenirken bir sorun oluştu.');
		}

		$mimeType = $this->mimeContentType($file['tmp_name']);
		if($mimeType == 'inode/x-empty' || $mimeType == 'application/octet-stream'){
			return array('error' => true , 'file' => $file , 'msg' => 'içi boş dosya yükleyemezsiniz.');
		}

		if (!in_array($mimeType, $mimetypes)) {
			return array('error' => true , 'file' => $file , 'msg' => 'Bu dosya türü yüklenemez.');
		}

		if($file['size'] > $this->maxUploadFileSize){
			return array('error' => true , 'file' => $file , 'msg' => sprintf('Yüklemek istediğiniz Dosya en fazla %s MB olabilir.', round($this->maxUploadFileSize, 2)));
		}

		$extension = array_search($mimeType, $mimetypes);
		if ($this->randomName) {
			$name = $this->randomFileName($file['name'], $file['size'], $file['tmp_name'], $extension);
		}else{
			$name = $this->filename($file['name'], $extension);
		}

		if (empty($name)) {
			return array('error' => true , 'file' => $file , 'msg' => 'Dosya adınız boş olamaz.');
		}

		$fullName       = $name . '.' . $extension;
		$fileUploadPath = $this->getUploadPath() . $fullName;

		if(!move_uploaded_file($file['tmp_name'], $fileUploadPath)) {
			return array('error' => true , 'file' => $file , 'msg' => 'Dosya taşıması başarısız oldu.');
		}

		$response = array(
			'fullname' => $fullName,
			'path'     => $this->getUploadFolder(),
			'images'   => array(
				'original' => $fullName
			)
		);

		if (isset($this->imageResize['small'])) {
			$response['images']['small'] = $this->imageMinimize($name, $extension, $fileUploadPath, $this->imageResize['small']);
		}

		if (isset($this->imageResize['medium'])) {
			$response['images']['medium'] = $this->imageMinimize($name, $extension, $fileUploadPath, $this->imageResize['medium']);
		}

		if (isset($this->imageResize['large'])) {
			$response['images']['large'] = $this->imageMinimize($name, $extension, $fileUploadPath, $this->imageResize['large']);
		}

		return array('error' => false, 'data' => $response);
	}

	private function imageMinimize($name, $extension, $filePath, $imageResize)
	{

		switch ($extension) 
		{
		  	case 'gif': $original = imagecreatefromgif($filePath); break;
		  	case 'png': $original = imagecreatefrompng($filePath); break;
		  	default:    $original = imagecreatefromjpeg($filePath); break;
		}


		$thumb_w = $imageResize['width'];
		$thumb_h = $imageResize['height'];
		list($original_w, $original_h) = getimagesize($filePath);

	    $thumb_w_resize = $thumb_w;
	    $thumb_h_resize = $thumb_h;
	    if ($original_w > $original_h)
	    {
	        $thumb_h_ratio  = $thumb_h / $original_h;
	        $thumb_w_resize = (int)round($original_w * $thumb_h_ratio);
	    }
	    else
	    {
	        $thumb_w_ratio  = $thumb_w / $original_w;
	        $thumb_h_resize = (int)round($original_h * $thumb_w_ratio);
	    }
	    if ($thumb_w_resize < $thumb_w)
	    {
	        $thumb_h_ratio  = $thumb_w / $thumb_w_resize;
	        $thumb_h_resize = (int)round($thumb_h * $thumb_h_ratio);
	        $thumb_w_resize = $thumb_w;
	    }

	    $thumb = imagecreatetruecolor($thumb_w_resize, $thumb_h_resize);

		imagecopyresampled($thumb, $original, 0, 0, 0, 0, $thumb_w_resize, $thumb_h_resize, $original_w, $original_h);

		$final = imagecreatetruecolor($thumb_w, $thumb_h);


	    $thumb_w_offset = 0;
	    $thumb_h_offset = 0;
	    if ($thumb_w < $thumb_w_resize)
	    {
	        $thumb_w_offset = (int)round(($thumb_w_resize - $thumb_w) / 2);
	    }
	    else
	    {
	        $thumb_h_offset = (int)round(($thumb_h_resize - $thumb_h) / 2);
	    }

	    imagecopy($final, $thumb, 0, 0, $thumb_w_offset, $thumb_h_offset, $thumb_w_resize, $thumb_h_resize);

		$newFileName = $name . '-' . $thumb_w . 'x' . $thumb_h . '.' . $extension;
		$newFilePath = $this->getUploadPath() . $newFileName;
		switch ($extension) 
		{
		  	case 'gif': imagegif($final, $newFilePath); break;
		  	case 'png': imagepng($final, $newFilePath); break;
		  	default: imagejpeg($final, $newFilePath, 80); break;
		}

		imagedestroy($final);
		imagedestroy($original);
		unset($final, $original);

		return $newFileName;
	}

}