<?php 

Class Database extends PDO
{

	/**
	 *
	 * Veritabanı bağlantı durumunu saklar.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var boolean $connection
	 *
	 */
	private $connection = false;

	/**
	 *
	 * Database Bilgileri
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @var array $database
	 *
	 */
	private $database;

	/**
	 *
	 * Veritabanı bağlantısını başlatır.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $dns
	 * @param string $username
	 * @param string $password
	 *
	 */
	public function __construct($host, $databasename, $username, $password)
	{
		try {

			$this->database = array('host' => $host, 'database' => $databasename, 'username' => $username, 'password' => $password);
	
			parent::__construct(sprintf('mysql:dbname=%s;host=%s', $databasename, $host), $username, $password);
	        $this->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
	        $this->setAttribute(PDO::MYSQL_ATTR_USE_BUFFERED_QUERY, true);

			$this->connection = true;
			
		} catch (Exception $e) {
			$this->connection = false;
		}
	}

	/**
	 *
	 * Veritabanı bağlantı durumunu döner
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @return boolean 
	 *
	 */
	public function isConnected()
	{
		return $this->connection;
	}

	/**
	 *
	 * Tabloyu kaldırır
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $tableName 
	 *
	 */
	public function dropTable($tableName)
	{
		$prepare = $this->prepare('DROP TABLE ' . $tableName);
		$prepare->execute(array());
	}

	/**
	 *
	 * Verileri içeri aktarır.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $tableName 
	 * @return boolean 
	 *
	 */
	public function importDatabase($fileName)
	{

		if (!file_exists($fileName)) {
			return false;
		}

	    $fp = fopen($fileName, 'r');

	    $templine = '';
	    while (($line = fgets($fp)) !== false) 
	    {

	        if (substr($line, 0, 2) == '--' || $line == '') {
	            continue;
	        }

	        $templine .= $line;

	        if (substr(trim($line), -1, 1) == ';') {
	            $this->query($templine);
	            $templine = '';
	        }
	    }

	    fclose($fp);
	    return true;
	}

	/**
	 *
	 * Metni json formatına çevirir.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $text
	 * @return string 
	 *
	 */
	protected function fileToJson($fileName)
	{
		return json_encode(json_decode(file_get_contents($fileName)));		
	}

	/**
	 *
	 * Admin ekler.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $email
	 * @param string $password
	 *
	 */
	public function addAdmin($email, $password)
	{
		$prepare = $this->prepare('INSERT INTO is_admins (email, password, register_date) VALUES(?, ?, ?)');
		$prepare->execute(array($email, $this->encryptPassword($password), time()));
	}

	/**
	 *
	 * Sayfa ekler.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $pageName
	 * @param string $pageData
	 *
	 */
	public function addPage($pageName, $fileName, $status = true)
	{
		$prepare = $this->prepare('INSERT INTO is_pages (pageName, pageData) VALUES(?, ?)');
		$prepare->execute(array($pageName, $this->fileToJson($fileName)));
	}

	/**
	 *
	 * Sistem özelliği ekler.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $optionName
	 * @param string $optionValue
	 * @param boolean $status
	 *
	 */
	public function addOption($optionName, $optionValue, $status = true)
	{



		$optionValue = str_replace('', '', $optionValue);

		$prepare = $this->prepare('INSERT INTO is_options (option_name, option_value, option_auto) VALUES(?, ?, ?)');
		$prepare->execute(array($optionName, $optionValue, $status === true));
	}

	/**
	 *
	 * Sistem özelliği ekler JSON Formatında.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $optionName
	 * @param string $optionValue
	 * @param boolean $status
	 *
	 */
	public function addOptionJson($optionName, $fileName, $data = array())
	{

		$responseData = json_decode(file_get_contents($fileName));
		if (isset($responseData->company->name) && isset($data['companyname'])) {
			$responseData->company->name = $data['companyname'];
		}

		$this->addOption($optionName, json_encode($responseData), true);
	}

	/**
	 *
	 * Config dosyasını oluşturur.
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $configPath
	 *
	 */
	public function createConfigFile($configPath)
	{

		$configResponse = file_get_contents(INSTALL_DATA_PATH . 'php/config.php');
		$configResponse = str_replace('{host}', $this->escapeString($this->database['host']), $configResponse);
        $configResponse = str_replace('{username}', $this->escapeString($this->database['username']), $configResponse);
        $configResponse = str_replace('{password}', $this->escapeString($this->database['password']), $configResponse);
        $configResponse = str_replace('{dbname}', $this->escapeString($this->database['database']), $configResponse);

		$handle = fopen($configPath, 'w+');
		fwrite($handle,$configResponse);
		fclose($handle);
		return true;
	}

	/**
	 *
	 * Dosyaya yazılacak metinler için özel karakterleri düzenler
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $text
	 * @return string 
	 *
	 */
	private function escapeString($text)
	{
		return str_replace('\'', '\\\'', str_replace('\\', '\\\\', $text));
	}

	/**
	 *
	 * Şifreleme algoritması
	 *
	 * @author Ismail Satilmis <ismaiil_0234@hotmail.com>
	 * @param string $text
	 * @return string 
	 *
	 */
	private function encryptPassword($text)
	{
		return md5('lambda' . sha1(md5(md5($text))) . '27.04.1999');
	}

}